% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/korsmeyer_peppas.R
\name{korsmeyer_peppas_model}
\alias{korsmeyer_peppas_model}
\title{Korsmeyer-Peppas Drug Release Kinetic Model}
\arguments{
\item{data}{A data frame containing experimental cumulative percent drug release.}

\item{time_col}{Character string specifying the column name for time (minutes).}

\item{release_col}{Character string specifying the column name for
cumulative percent drug release.}

\item{group_col}{Optional character string specifying a grouping variable
(e.g., formulation, batch).}

\item{pH_col}{Optional character string specifying a column containing pH values.}

\item{plot}{Logical; if TRUE, generates a plot with fitted Korsmeyer-Peppas lines.}

\item{annotate}{Logical; if TRUE, annotates the plot with kKP, n, intercept,
R^2, and t50 (only if <= 2 groups).}

\item{normalize}{Logical; if TRUE (default), normalizes cumulative percent release
to fraction (0-1). If FALSE, retains original percent values.}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{Data frame with kKP, n, intercept, R^2, and t50 for each group.}
  \item{\code{data}}{Processed data used for fitting and plotting.}
}
}
\description{
Fits experimental cumulative drug release data to the Korsmeyer-Peppas model
using log10-transformed fraction released vs. log10-transformed time. The function
automatically normalizes cumulative percent drug release to fraction (0-1) by default
and removes t = 0. In addition, the function supports optional grouping variables
(e.g., formulation, batch) and optional pH-dependent analysis. It generates
publication-quality plots with experimental curves, fitted Korsmeyer-Peppas
straight lines, mechanism table, and annotations for Korsmeyer-Peppas release
constant (kKP), release exponent (n), intercept, coefficient of determination
(R^2), and the time required for 50-percent drug release (t50).

Users can toggle 'normalize = TRUE/FALSE' to use fraction (0-1) or raw percent drug release.

Model:
log10(Mt/MInf) = log10(k) + n * log10(t)

The release exponent, n, indicates the drug release mechanism:
     - n = 0.5 : Fickian diffusion
     - 0.5 < n < 1 : Non-Fickian (anomalous) diffusion
     - n = 1   : Case II transport (zero-order release)
     - n > 1   : Super Case II transport
}
\examples{
# Example I: Single formulation
df1 <- data.frame(
  time = c(0, 15, 30, 45, 60, 90, 120, 150, 180),
  release = c(0, 11.4, 20.8, 30.8, 39.8, 57.8, 72, 84.8, 93.5)
)
korsmeyer_peppas_model(
  data = df1,
  time_col = "time",
  release_col = "release",
  normalize = TRUE  # default
)

# Example II: Two formulations (grouped, not pH-dependent)
df2 <- data.frame(
  time = rep(c(0, 30, 60, 90, 120, 150), 2),
  release = c(
    0, 18, 35, 55, 72, 88,  # Formulation A
    0, 12, 26, 40, 58, 70   # Formulation B
  ),
  formulation = rep(c("Formulation A", "Formulation B"), each = 6)
)
korsmeyer_peppas_model(
  data = df2,
  time_col = "time",
  release_col = "release",
  group_col = "formulation"
)

# Example III: pH-dependent release
df_pH <- data.frame(
  time = rep(c(0, 60, 120, 180), 2),
  release = c(0, 40, 75, 95, 0, 30, 60, 80),
  pH = rep(c(7.4, 4.5), each = 4)
)
korsmeyer_peppas_model(
  data = df_pH,
  time_col = "time",
  release_col = "release",
  pH_col = "pH"
)

# Example IV: Two formulations under two pH conditions
df1 <- data.frame(
  time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
  release = c(
    0, 12, 25, 38, 50, 62, 73, 82, 89, 95,  # pH 4.5
    0, 15, 30, 45, 59, 70, 79, 86, 91, 97   # pH 7.6
  ),
  pH = rep(c(4.5, 7.6), each = 10)
)
df2 <- data.frame(
  time = rep(c(0, 15, 30, 45, 60, 75, 90, 105, 120, 135), 2),
  release = c(
    0, 10, 22, 34, 46, 57, 67, 76, 84, 91,  # pH 4.5
    0, 13, 27, 41, 55, 67, 77, 85, 92, 98   # pH 7.6
  ),
  pH = rep(c(4.5, 7.6), each = 10)
)
df_all <- rbind(
  cbind(formulation = "Dataset 1", df1),
  cbind(formulation = "Dataset 2", df2)
)
korsmeyer_peppas_model(
  data = df_all,
  time_col = "time",
  release_col = "release",
  group_col = "formulation",
  pH_col = "pH"
)
}
\references{
Korsmeyer, R. W., Gurny, R., Doelker, E., Buri, P., & Peppas, N. A.
(1983) <doi:10.1016/0378-5173(83)90064-9> Mechanisms of solute release from
porous hydrophilic polymers. International Journal of Pharmaceutics, 15(1), 25–35.
}
\author{
Paul Angelo C. Manlapaz
}
