% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rprocess_spec.R
\docType{methods}
\name{rprocess_spec}
\alias{rprocess_spec}
\alias{onestep}
\alias{discrete_time}
\alias{euler}
\alias{gillespie}
\alias{gillespie_hl}
\title{rprocess specification}
\usage{
onestep(step.fun)

discrete_time(step.fun, delta.t = 1)

euler(step.fun, delta.t)

gillespie(rate.fun, v, hmax = Inf)

gillespie_hl(..., .pre = "", .post = "", hmax = Inf)
}
\arguments{
\item{step.fun}{a C snippet, an R function, or
the name of a native routine in a shared-object library.
This gives a procedure by which one simulates a single step of the latent state process.}

\item{delta.t}{positive numerical value; for \code{euler} and \code{discrete_time}, the size of the step to take}

\item{rate.fun}{a C snippet, an R function, or
the name of a native routine in a shared-object library.
This gives a procedure by which one computes the event-rate of the elementary events in the continuous-time latent Markov chain.}

\item{v}{integer matrix; giving the stoichiometry of the continuous-time latent Markov process.
It should have dimensions \code{nvar} x \code{nevent}, where \code{nvar} is the number of state variables and \code{nevent} is the number of elementary events.
\code{v} describes the changes that occur in each elementary event:
it will usually comprise the values 1, -1, and 0 according to whether a state variable is incremented, decremented, or unchanged in an elementary event.
The rows of \code{v} may be unnamed or named.
If the rows are unnamed, they are assumed to be in the same order as the vector of state variables returned by \code{rinit}.
If the rows are named,
the names of the state variables returned by \code{rinit} will be matched
to the rows of \code{v} to ensure a correct mapping.
If any of the row names of \code{v} cannot be found among the state variables or if any row names of \code{v} are duplicated, an error will occur.}

\item{hmax}{maximum time step allowed (see below)}

\item{\dots}{individual C snippets corresponding to elementary events}

\item{.pre, .post}{C snippets (see Details)}
}
\description{
Specification of the latent state process simulator, rprocess.
}
\section{Discrete-time processes}{

If the state process evolves in discrete time, specify \code{rprocess} using the \code{discrete_time} plug-in.
Specifically, provide
\preformatted{
    rprocess = discrete_time(step.fun = f, delta.t),
}
where \code{f} is a C snippet or \R function that simulates one step of the state process.
The former is the preferred option, due to its much greater computational efficiency.
The goal of such a C snippet is to replace the state variables with their new random values at the end of the time interval.
Accordingly, each state variable should be over-written with its new value.
In addition to the states, parameters, covariates (if any), and observables, the variables \code{t} and \code{dt}, containing respectively the time at the beginning of the step and the step's duration, will be defined in the context in which the C snippet is executed.
See \code{\link{Csnippet}} for general rules on writing C snippets.
Examples are to be found in the tutorials on the \href{https://kingaa.github.io/pomp/}{package website}.

If \code{f} is given as an \R function, its arguments should come from the state variables, parameters, covariates, and time.
It may also take the argument \sQuote{\code{delta.t}};
when called, the latter will be the timestep.
It must also have the argument \sQuote{\code{...}}.
It should return a named vector of length equal to the number of state variables, representing a draw from the distribution of the state process at time \code{t+delta.t} conditional on its value at time \code{t}.
}

\section{Continuous-time processes}{

If the state process evolves in continuous time, but you can use an Euler approximation, implement \code{rprocess} using the \code{euler} plug-in.
Specify
\preformatted{
    rprocess = euler(step.fun = f, delta.t)
}
in this case.
As before, \code{f} can be provided either as a C snippet or as an \R function, the former resulting in much quicker computations.
The form of \code{f} will be the same as above (in the discrete-time case).

If you have a procedure that allows you, given the value of the state process at any time,
to simulate it at an arbitrary time in the future, use the \code{onestep} plug-in.
To do so, specify
\preformatted{
    rprocess = onestep(step.fun = f).
}
Again, \code{f} can be provided either as a C snippet or as an \R function, the former resulting in much quicker computations.
The form of \code{f} should be as above (in the discrete-time or Euler cases).
}

\section{Size of time step}{

The simulator plug-ins \code{discrete_time}, \code{euler}, and \code{onestep} all work by taking discrete time steps.
They differ as to how this is done.
Specifically,
\enumerate{
\item \code{onestep} takes a single step to go from any given time \code{t1} to any later time \code{t2} (\code{t1 <= t2}).
Thus, this plug-in is designed for use in situations where a closed-form solution to the process exists.
\item To go from \code{t1} to \code{t2}, \code{euler} takes \code{n} steps of equal size, where
\preformatted{
    n = ceiling((t2-t1)/delta.t).
}
\item \code{discrete_time} assumes that the process evolves in discrete time, where the interval between successive times is \code{delta.t}.
Thus, to go from \code{t1} to \code{t2}, \code{discrete_time} takes \code{n} steps of size exactly \code{delta.t}, where
\preformatted{
    n = floor((t2-t1)/delta.t).
}
}
}

\section{Exact (event-driven) simulations}{

If you desire exact simulation of certain continuous-time Markov chains, an implementation of Gillespie's algorithm (Gillespie 1977) is available,
via the \code{gillespie} and \code{gillespie_hl} plug-ins.
The former allows for the rate function to be provided as an \R function or a single C snippet,
while the latter provides a means of specifying the elementary events via a list of C snippets.

A high-level interface to the simulator is provided by \code{gillespie_hl}.
To use it, supply
\preformatted{
    rprocess = gillespie_hl(..., .pre = "", .post = "", hmax = Inf)
}
to \code{pomp}.
Each argument in \code{...} corresponds to a single elementary event and should be a list containing two elements.
The first should be a string or C snippet;
the second should be a named integer vector.
The variable \code{rate} will exist in the context of the C snippet, as will the parameter, state variables, covariates, and the time \code{t}.
The C snippet should assign to the variable \code{rate} the corresponding elementary event rate.

The named integer vector specifies the changes to the state variables corresponding to the elementary event.
There should be named value for each of the state variables returned by \code{rinit}.
The arguments \code{.pre} and \code{.post} can be used to provide C code that will run respectively before and after the elementary-event snippets.
These hooks can be useful for avoiding duplication of code that performs calculations needed to obtain several of the different event rates.

Here's how a simple birth-death model might be specified:
\preformatted{
    gillespie_hl(
        birth=list("rate = b*N;",c(N=1)),
        death=list("rate = m*N;",c(N=-1))
    )
}
In the above, the state variable \code{N} represents the population size and parameters \code{b}, \code{m} are the birth and death rates, respectively.

To use the lower-level \code{gillespie} interface, furnish
\preformatted{
    rprocess = gillespie(rate.fun = f, v, hmax = Inf)
}
to \code{pomp}, where \code{f} gives the rates of the elementary events.
Here, \code{f} may be furnished as an \R function or as a C snippet.
If \code{f} is an \R function, its arguments should come from the state variables, parameters, covariates, and time.
It must also have the arguments \sQuote{\code{j}} and \sQuote{\code{...}}.
When \code{f} is called,
the integer \code{j} will indicate the elementary event (corresponding to the column the matrix \code{v}, see below).
\code{f} should return a single numerical value, representing the rate of that elementary event at that point in state space and time.
If \code{f} is supplied as a C snippet, the parameters, latent state variables, covariates, and time will be visible in the context wherein the snippet is executed, as will the integer \sQuote{\code{j}}.
The purpose of the C snippet is to fill the double-precision variable \sQuote{\code{rate}} with the corresponding event rate.

Here, the stoichiometric matrix \code{v} specifies the continuous-time Markov process in terms of its elementary events.
It should have dimensions \code{nvar} x \code{nevent}, where \code{nvar} is the number of state variables and \code{nevent} is the number of elementary events.
\code{v} describes the changes that occur in each elementary event:
it will usually comprise the values 1, -1, and 0 according to whether a state variable is incremented, decremented, or unchanged in an elementary event.
The rows of \code{v} should have names corresponding to the state variables.
If any of the row names of \code{v} cannot be found among the state variables or if any row names of \code{v} are duplicated, an error will occur.

This lower-level interface may be preferable if it is easier to write code that calculates the correct rate based on \code{j} rather than to write a snippet for each possible value of \code{j}.
For example, if the number of possible values of \code{j} is large and the rates vary according to a few simple rules, the lower-level interface may provide the easier way of specifying the model.

When the process is non-autonomous (i.e., the event rates depend explicitly on time), it can be useful to set \code{hmax} to the maximum step that will be taken.
By default, the elementary event rates will be recomputed at least once per observation interval.
}

\section{Default behavior}{

The default \code{rprocess} is undefined.
It will yield missing values (\code{NA}) for all state variables.
}

\section{Note for Windows users}{


Some Windows users report problems when using C snippets in parallel computations.
These appear to arise when the temporary files created during the C snippet compilation process are not handled properly by the operating system.
To circumvent this problem, use the \code{\link[=pomp]{cdir}} and \code{\link[=pomp]{cfile}} options to cause the C snippets to be written to a file of your choice, thus avoiding the use of temporary files altogether.
}

\seealso{
\code{\link{rprocess}}

More on implementing POMP models:
\code{\link{Csnippet}},
\code{\link{accumvars}},
\code{\link{basic_components}},
\code{\link{betabinomial}},
\code{\link{covariates}},
\code{\link{dinit_spec}},
\code{\link{dmeasure_spec}},
\code{\link{dprocess_spec}},
\code{\link{emeasure_spec}},
\code{\link{eulermultinom}},
\code{\link{parameter_trans}()},
\code{\link{pomp-package}},
\code{\link{pomp_constructor}},
\code{\link{prior_spec}},
\code{\link{rinit_spec}},
\code{\link{rmeasure_spec}},
\code{\link{skeleton_spec}},
\code{\link{transformations}},
\code{\link{userdata}},
\code{\link{vmeasure_spec}}
}
\concept{implementation information}
