#' @title Main function to simulate expected yield using the Beverton-Holt Yield Per Recruit model for a slot limit
#'
#' @description Main wrapper function to estimate yield using the Beverton-Holt YPR model. This main function accepts a range of values for cf, cm, recruitment length, lower slot limit length, and upper slot limit length.
#'
#' @details Details will be filled out later
#'
#' @param recruitmentTL A numeric representing the minimum length limit for recruiting to the fishery in mm.
#' @param lowerSL A numeric representing the length of the lower slot limit in mm.
#' @param upperSL A numeric representing the length of the upper slot limit in mm.
#' @param cfunder Single value, conditional fishing mortality under the lower slot limit.
#' @param cfin Single value, conditional fishing mortality within the lower and upper slot limit.
#' @param cfabove Single value, conditional fishing mortality over the upper slot limit.
#' @param cmmin Single value, minimum conditional natural mortality
#' @param cmmax Single value, maximum conditional natural mortality
#' @param cminc Single value, increment to cycle from minimum to maximum conditional natural mortality
#' @param loi A numeric vector for lengths of interest. Used to determine number of fish that reach desired lengths.
#' @param lhparms A named vector or list that contains values for each `N0`, `tmax`, `Linf`, `K`, `t0`, `LWalpha`, and `LWbeta`. See \code{\link{makeLH}} for definitions of these life history parameters. Also see details.
#' @param matchRicker A logical that indicates whether the yield function should match that in Ricker (). Defaults to \code{TRUE}. The only reason to changed to \code{FALSE} is to try to match output from FAMS. See the "YPR_FAMSvRICKER" article.
#'
#' @return A data.frame with the following calculated values:
#' \itemize{
#' \item cm A numeric representing conditional natural mortality
#' \item TotalYield is the calculated total yield
#' \item TotalHarvest is the calculated total number of harvested fish
#' \item TotalNdie is the calculated total number of fish that die of natural death
#' \item yieldUnder is the calculated yield under the slot limit
#' \item yieldIn is the calculated yied within the slot limit
#' \item yieldAbove is the calculated yield above the slot limit
#' \item exploitationUnder is the exploitation rate under the slot limit
#' \item exploitationIn is the exploitation rate within the slot limit
#' \item exploitationAbove is the exploitation rate above the slot limit
#' \item NharvestUnder is the number of harvested fish under the slot limit
#' \item NharvestIn is the number of harvested fish within the slot limit
#' \item NharvestAbove is the number of harvested fish above the slot limit
#' \item NdieUnder is the number of fish that die of natural death under the slot limit
#' \item NdieIn is the number of fish that die of natural deaths within the slot limit
#' \item NdieAbove is the number of fish that die of natural deaths above the slot limit
#' \item avglenUnder is the average length of fish harvested under the slot limit
#' \item avglenIn is the average length of fish harvested within the slot limit
#' \item avglenAbove is the average length of fish harvested above the slot limit
#' \item avgwtUnder is the average weight of fish harvested under the slot limit
#' \item avgwtIn is the average weight of fish harvested within the slot limit
#' \item avgwtAbove is the average weight of fish harvested above the slot limit
#' \item trUnder is the time for a fish to recruit to a minimum length limit (i.e., time to enter fishery)
#' \item trIn is the time for a fish to recruit to a lower length limit of the slot limit
#' \item trOver is the time for a fish to recruit to a upper length limit of the slot limit
#' \item NtUnder is the number of fish at time trUnder (time they become harvestable size under the slot limit)
#' \item NtIn is the number of fish at time trIn (time they reach the lower slot limit size)
#' \item NtAbove is the number of fish at time trAbove (time they reach the upper slot limit size)
#' \item FUnder is the estimated instantaneous rate of fishing mortality under the slot limit
#' \item FIn is the estimated instantaneous rate of fishing mortality within the slot limit
#' \item FAbove is the estimated instantaneous rate of fishing mortality above the slot limit
#' \item MUnder is the estimated  instantaneous rate of natural mortality under the slot limit
#' \item MIn is the estimated  instantaneous rate of natural mortality within the slot limit
#' \item MAbove is the estimated  instantaneous rate of natural mortality above the slot limit
#' \item ZUnder is the estimated  instantaneous rate of total mortality under the slot limit
#' \item ZIn is the estimated  instantaneous rate of total mortality within the slot limit
#' \item ZAbove is the estimated  instantaneous rate of total mortality above the slot limit
#' \item SUnder is the estimated total survival under the slot limit
#' \item SIn is the estimated total survival within the slot limit
#' \item SAbove is the estimated total survival above the slot limit
#' \item cfUnder A numeric representing conditional fishing mortality
#' \item cfIn A numeric representing conditional fishing mortality
#' \item cfOver A numeric representing conditional fishing mortality
#' \item recruitmentTL A numeric representing the minimum length limit for recruiting to the fishery in mm.
#' \item lowerSL A numeric representing the length of the lower slot limit in mm.
#' \item upperSL A numeric representing the length of the upper slot limit in mm.
#' \item N0 A numeric representing the initial number of new recruits entering the fishery OR a vector or list that contains named values for each \code{N0}, \code{Linf}, \code{K}, \code{t0}, \code{LWalpha}, \code{LWbeta}, and \code{tmax}
#' \item Linf A numeric representing the point estimate of the asymptotic mean length (L-infinity) from the von Bertalanffy growth model in mm
#' \item K A numeric representing the point estimate of the Brody growth coefficient from the von Bertalanffy growth model
#' \item t0 A numeric representing the point estimate of the x-intercept (i.e., theoretical age at a mean length of 0) from the von Bertalanffy growth model
#' \item LWalpha A numeric representing the point estimate of alpha from the length-weight regression on the log10 scale.
#' \item LWbeta A numeric representing the point estimate of beta from the length-weight regression on the log10 scale.
#' \item tmax An integer representing maximum age in the population in years
#' \item \code{N at xxx mm} is the number that reach the length of interest supplied. There will be one column for each length of interest.
#' #' }
#'
#' @author Jason C. Doll, \email{jason.doll@fmarion.edu}
#'
#' @examples
#' #Load other required packages for organizing output and plotting
#' library(ggplot2)
#' library(dplyr)
#' library(tidyr)
#'
#' # Custom theme for plots (to make look nice)
#' theme_FAMS <- function(...) {
#'   theme_bw() +
#'   theme(
#'     panel.grid.major=element_blank(),panel.grid.minor=element_blank(),
#'     axis.text=element_text(size=14,color="black"),
#'     axis.title=element_text(size=16,color="black"),
#'     axis.title.y=element_text(angle=90),
#'     axis.line=element_line(color="black"),
#'     panel.border=element_blank()
#'   )
#' }
#'
#' # Life history parameters to be used below
#' LH <- makeLH(N0=100,tmax=15,Linf=592,K=0.20,t0=-0.3,LWalpha=-5.528,LWbeta=3.273)
#'
#' #Estimate yield
#'  Res_1 <- yprBH_SlotLL(recruitmentTL=200,lowerSL=250,upperSL=325,
#'                        cfunder=0.25,cfin=0.6,cfabove=0.15,cmmin=0.3,cmmax=0.6,cminc=0.05,
#'                        loi=c(200,250,300,325,350),lhparms=LH)
#'
#'  Res_1
#'
#' # Plot results
#' # Total Yield vs Conditional Natural Mortality (cm)
#' ggplot(data=Res_1,mapping=aes(x=cm,y=TotalYield)) +
#'   geom_point() +
#'   geom_line() +
#'   labs(y="Total Yield (g)",x="Conditional Natural Mortality (cm)") +
#'   theme_FAMS()
#'
#'
#' # Yield under, in, and above the slot limit vs Conditional Natural Mortality (cm)
#' # Select columns for plotting
#' plot_data <- Res_1 %>%
#'   select(cm, yieldUnder, yieldIn, yieldAbove) %>%
#'   pivot_longer(!cm, names_to="YieldCat",values_to="Yield")
#'
#' # Generate plot
#' ggplot(data=plot_data,mapping=aes(x=cm,y=Yield,group=YieldCat,color=YieldCat)) +
#'   geom_point() +
#'   scale_color_discrete(name="Yield",labels=c("Above SL","In SL","Under SL"))+
#'   geom_line() +
#'   labs(y="Total Yield (g)",x="Conditional Natural Mortality (cm)") +
#'   theme_FAMS() +
#'   theme(legend.position = "top")+
#'   guides(color=guide_legend(title="Yield"))
#'
#'
#' @rdname yprBH_SlotLL.R
#' @export
yprBH_SlotLL<-function(recruitmentTL,lowerSL,upperSL,cfunder,cfin,cfabove,cmmin,cmmax,cminc,
                       loi=NA,lhparms,matchRicker=FALSE){

  # ---- Check inputs
  iCheckrecruitTL(recruitmentTL)
  iChecklowerSLTL(lowerSL)
  iCheckupperSLTL(upperSL)
  iCheckslotOrder(recruitmentTL, lowerSL, upperSL)
  iCheckLLinf(recruitmentTL,lhparms$Linf)
  iCheckLLinf(lowerSL,lhparms$Linf)
  iCheckLLinf(upperSL,lhparms$Linf)
  iCheckcfunder(cfunder)
  iCheckcfin(cfin)
  iCheckcfabove(cfabove)
  iCheckcm(cmmin,"minimum")
  iCheckcm(cmmax,"maximum")
  cm <- iCheckcfminc(cminc,cmmin,cmmax)
  iCheckloi(loi)

  # Setup data.frame of input values (varying cf and cm, the rest constant)
  res <- expand.grid(recruitmentTL=recruitmentTL,lowerSL=lowerSL,upperSL=upperSL,
                     cfunder=cfunder,cfin=cfin,cfabove=cfabove,
                     cm=seq(cmmin,cmmax,cminc))
  # Send each row to ypr_func() ... so calc yield et al for all cf & cm combos
  # output is by age
  res <- purrr::pmap_df(res,yprBH_slot_func,matchRicker=matchRicker,loi=loi,lhparms=lhparms)

  # Return result
  return(res)

}


