% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/expect-condition.R
\name{expect_error}
\alias{expect_error}
\alias{expect_warning}
\alias{expect_message}
\alias{expect_condition}
\title{Do you expect an error, warning, message, or other condition?}
\usage{
expect_error(
  object,
  regexp = NULL,
  class = NULL,
  ...,
  inherit = TRUE,
  info = NULL,
  label = NULL
)

expect_warning(
  object,
  regexp = NULL,
  class = NULL,
  ...,
  inherit = TRUE,
  all = FALSE,
  info = NULL,
  label = NULL
)

expect_message(
  object,
  regexp = NULL,
  class = NULL,
  ...,
  inherit = TRUE,
  all = FALSE,
  info = NULL,
  label = NULL
)

expect_condition(
  object,
  regexp = NULL,
  class = NULL,
  ...,
  inherit = TRUE,
  info = NULL,
  label = NULL
)
}
\arguments{
\item{object}{Object to test.

Supports limited unquoting to make it easier to generate readable failures
within a function or for loop. See \link{quasi_label} for more details.}

\item{regexp}{Regular expression to test against.
\itemize{
\item A character vector giving a regular expression that must match the
error message.
\item If \code{NULL}, the default, asserts that there should be an error,
but doesn't test for a specific value.
\item If \code{NA}, asserts that there should be no errors, but we now recommend
using \code{\link[=expect_no_error]{expect_no_error()}} and friends instead.
}

Note that you should only use \code{message} with errors/warnings/messages
that you generate. Avoid tests that rely on the specific text generated by
another package since this can easily change. If you do need to test text
generated by another package, either protect the test with \code{skip_on_cran()}
or use \code{expect_snapshot()}.}

\item{class}{Instead of supplying a regular expression, you can also supply
a class name. This is useful for "classed" conditions.}

\item{...}{
  Arguments passed on to \code{\link[=expect_match]{expect_match}}
  \describe{
    \item{\code{fixed}}{If \code{TRUE}, treats \code{regexp} as a string to be matched exactly
(not a regular expressions). Overrides \code{perl}.}
    \item{\code{perl}}{logical.  Should Perl-compatible regexps be used?}
  }}

\item{inherit}{Whether to match \code{regexp} and \code{class} across the
ancestry of chained errors.}

\item{info}{Extra information to be included in the message. This argument
is soft-deprecated and should not be used in new code. Instead see
alternatives in \link{quasi_label}.}

\item{label}{Used to customise failure messages. For expert use only.}

\item{all}{\emph{DEPRECATED} If you need to test multiple warnings/messages
you now need to use multiple calls to \code{expect_message()}/
\code{expect_warning()}}
}
\value{
If \code{regexp = NA}, the value of the first argument; otherwise
the captured condition.
}
\description{
\code{expect_error()}, \code{expect_warning()}, \code{expect_message()}, and
\code{expect_condition()} check that code throws an error, warning, message,
or condition with a message that matches \code{regexp}, or a class that inherits
from \code{class}. See below for more details.

In the 3rd edition, these functions match (at most) a single condition. All
additional and non-matching (if \code{regexp} or \code{class} are used) conditions
will bubble up outside the expectation. If these additional conditions
are important you'll need to catch them with additional
\code{expect_message()}/\code{expect_warning()} calls; if they're unimportant you
can ignore with \code{\link[=suppressMessages]{suppressMessages()}}/\code{\link[=suppressWarnings]{suppressWarnings()}}.

It can be tricky to test for a combination of different conditions,
such as a message followed by an error. \code{\link[=expect_snapshot]{expect_snapshot()}} is
often an easier alternative for these more complex cases.
}
\section{Testing \code{message} vs \code{class}}{

When checking that code generates an error, it's important to check that the
error is the one you expect. There are two ways to do this. The first
way is the simplest: you just provide a \code{regexp} that match some fragment
of the error message. This is easy, but fragile, because the test will
fail if the error message changes (even if its the same error).

A more robust way is to test for the class of the error, if it has one.
You can learn more about custom conditions at
\url{https://adv-r.hadley.nz/conditions.html#custom-conditions}, but in
short, errors are S3 classes and you can generate a custom class and check
for it using \code{class} instead of \code{regexp}.

If you are using \code{expect_error()} to check that an error message is
formatted in such a way that it makes sense to a human, we recommend
using \code{\link[=expect_snapshot]{expect_snapshot()}} instead.
}

\examples{
# Errors ------------------------------------------------------------------
f <- function() stop("My error!")
expect_error(f())
expect_error(f(), "My error!")

# You can use the arguments of grepl to control the matching
expect_error(f(), "my error!", ignore.case = TRUE)

# Note that `expect_error()` returns the error object so you can test
# its components if needed
err <- expect_error(rlang::abort("a", n = 10))
expect_equal(err$n, 10)

# Warnings ------------------------------------------------------------------
f <- function(x) {
  if (x < 0) {
    warning("*x* is already negative")
    return(x)
  }
  -x
}
expect_warning(f(-1))
expect_warning(f(-1), "already negative")
expect_warning(f(1), NA)

# To test message and output, store results to a variable
expect_warning(out <- f(-1), "already negative")
expect_equal(out, -1)

# Messages ------------------------------------------------------------------
f <- function(x) {
  if (x < 0) {
    message("*x* is already negative")
    return(x)
  }

  -x
}
expect_message(f(-1))
expect_message(f(-1), "already negative")
expect_message(f(1), NA)
}
\seealso{
\code{\link[=expect_no_error]{expect_no_error()}}, \code{expect_no_warning()},
\code{expect_no_message()}, and \code{expect_no_condition()} to assert
that code runs without errors/warnings/messages/conditions.

Other expectations: 
\code{\link{comparison-expectations}},
\code{\link{equality-expectations}},
\code{\link{expect_length}()},
\code{\link{expect_match}()},
\code{\link{expect_named}()},
\code{\link{expect_null}()},
\code{\link{expect_output}()},
\code{\link{expect_reference}()},
\code{\link{expect_silent}()},
\code{\link{inheritance-expectations}},
\code{\link{logical-expectations}}
}
\concept{expectations}
